const express = require("express");
const mysql = require("mysql2/promise");
const cors = require("cors");
const { format } = require("date-fns");
const bodyParser = require("body-parser");
const https = require("https");
const fs = require("fs");
const Presale = require("./src/models/Presale");
const Sanitizer = require("./src/utils/Sanitizer");
const Role = require("./src/models/Role");

require("dotenv").config({ path: __dirname + "/.env" });

const privateKey = fs.readFileSync(__dirname + "/private.key", "utf-8");
const certificate = fs.readFileSync(__dirname + "/public.cert", "utf-8");

const options = {
  key: privateKey,
  cert: certificate,
};

// Crear la aplicación Express
const app = express();

app.use(bodyParser.urlencoded({ extended: false }));
app.use(bodyParser.json());

// Crear conexión a la base de datos
const connection = mysql.createPool({
  host: process.env.HOSTNAME,
  port: "3306",
  user: process.env.DBUSER,
  password: process.env.PASSWORD,
  database: process.env.DATABASE,
});

// Middleware para habilitar CORS
app.use(cors());

app.post("/api/registroPreventa", async (req, res) => {
  try {
    const {
      usuario,
      nameClient,
      job,
      decisions,
      phone,
      mail,
      companyName,
      address,
      bussinesDescription,
      problem,
      budgetIP,
      solution,
      brands,
      brandPreference,
      infrastructureIP,
      additionalInfo,
      link_folder,
      estatus,
    } = req.body;

    const query = `
      INSERT INTO ip_preventa (
        usuario,
        marca_temporal,
        nombre_cliente,
        puesto,
        decisiones,
        telefono,
        correo,
        nombre_empresa,
        direccion,
        descripcion_negocio,
        problema,
        presupuesto,
        solucion,
        marcas,
        preferencia_marcas,
        infraestructura_ti,
        informacion_adicional,
        link_carpeta,
        estatus
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
      `;

    const values = [
      usuario,
      new Date(),
      nameClient,
      job,
      decisions,
      phone,
      mail,
      companyName,
      address,
      bussinesDescription,
      problem,
      budgetIP,
      solution,
      brands,
      brandPreference,
      infrastructureIP,
      additionalInfo,
      link_folder,
      estatus,
    ];
    await connection.query(query, values);

    res.status(201).json({ message: "Rigistro guardado exitosamente" });
  } catch (error) {
    console.error("Error al guardar el registro", error);
    res.status(500).json({ error: "Error al guardar el registro" });
  }
});

app.post("/api/actualizarObservacionesIP", async (req, res) => {
  try {
    const { id, observacionesIP } = req.body;

    console.log("Id:", id);
    console.log("Observaciones1:", observacionesIP);
    console.log("Observaciones2:", observacionesIP);

    const query = "UPDATE ip_preventa SET observaciones = ? WHERE id = ?";

    const values = [observacionesIP, id];
    await connection.query(query, values);

    res.status(200).json({
      message: "Observaciones actualizadas exitosamente",
    });
  } catch (error) {
    console.error("Error al actualizar las observaciones", error);
    res.status(500).json({
      error: "Error al actualizar las observaciones",
    });
  }
});

const getUniqueValues = async (columnName, alias, req, withJoin = false) => {
  let uniqueValues = [];

  try {
    let { page = 1, limit = 20 } = req.query;
    limit = parseInt(limit);
    page = parseInt(page);
    const whereConstraint = Presale.whereConstraint(req);

    const offset = page <= 1 ? 0 : limit * (page - 1);

    whereConstraint.values.push(limit, offset);

    let query = `SELECT DISTINCT(master.${columnName}) AS ${alias} FROM (${whereConstraint.query} ${whereConstraint.dateRange}  LIMIT ? OFFSET ?) AS master`;

    if (withJoin) {
      query = `SELECT DISTINCT(master.${columnName}) AS value, (CASE WHEN master.${columnName} = u.id THEN u.name ELSE "Vacíos" END) AS label
        FROM (${whereConstraint.query} ${whereConstraint.dateRange}  LIMIT ? OFFSET ?) AS master
        LEFT JOIN users AS u ON master.${columnName} = u.id`;
    }

    const [rows] = await connection.query(query, whereConstraint.values);

    for (const row of rows) {
      if (withJoin) {
        uniqueValues.push(row);
        continue;
      }

      if (row[alias] == null || row[alias] == undefined) {
        uniqueValues.push('Vacíos');
      } else {
        uniqueValues.push(row[alias]);
      }
    }
  } catch (error) {
    console.error('getUniqueValues', error);
  }

  return uniqueValues;
}

const getInitialYear = async () => {
  try {
    const query = "SELECT created_at FROM ip_preventa LIMIT 1";
    const [rows] = await connection.query(query);

    const createdAt = rows[0].created_at;
    if (createdAt != undefined || createdAt.length > 10) {
      let initialYear = createdAt.getFullYear();
      const currentYear = new Date();

      if (initialYear == currentYear.getFullYear()) {
        return ["Todos", `${initialYear}`];
      }

      let years = [];
      while (initialYear <= currentYear.getFullYear()) {
        years.push(`${initialYear}`);
        initialYear++;
      }

      years.unshift("Todos");

      return years;
    }
  } catch (error) {
    console.error("getInitialDate", error);
  }

  return [];
};

app.get("/api/preventa/filtros", async (req, res) => {
  let filtros = {
    filtroVendedor: [],
    filtroSeguidorAsignado: [],
    filtroAnios: [],
  };

  try {
    const filtroVendedor = await getUniqueValues('user_id', 'name', req, true);
    const filtroSeguidorAsignado = await getUniqueValues('designated_user_id', 'name', req, true);
    const initialYear = await getInitialYear();

    filtros.filtroVendedor = filtroVendedor;
    filtros.filtroSeguidorAsignado = filtroSeguidorAsignado;
    filtros.filtroAnios = initialYear;

    res.status(200).json(filtros);
  } catch (error) {
    console.error("Error al obtener los filtros", error);
    res.status(500).json({ error: "Error al obtener los filtros" });
  }
});

app.get("/api/consultaPreventa", async (req, res) => {
  try {
    let { page = 1, limit = 20 } = req.query;
    limit = parseInt(limit);
    page = parseInt(page);
    const whereConstraint = Presale.whereConstraint(req);

    const offset = page <= 1 ? 0 : limit * (page - 1);

    let query = whereConstraint.query;

    const [items] = await connection.query(
      `SELECT COUNT(*) AS rows_count FROM (${query} ${whereConstraint.dateRange}) AS tblCounter`,
      whereConstraint.values
    );

    let totalPages = 1;
    const rowsCount = items[0];

    const totalItems = parseInt(rowsCount.rows_count);

    if (totalItems > limit) {
      let doubleRes = totalItems / limit;
      let intResult = Math.round(doubleRes);

      totalPages = intResult > doubleRes ? intResult : intResult + 1;
    }

    const prevPage =
      totalItems > limit ? (page >= 2 ? page - 1 : null) : null;
    const nextPage =
      totalItems > limit ? (page < totalPages ? page + 1 : null) : null;

    whereConstraint.values.push(limit, offset);

    query = `
      SELECT master.*, (CASE WHEN master.user_id = users.id
        THEN users.name
        ELSE null
        END) AS vendedor,
        (SELECT users.name AS seguidor FROM users WHERE users.id = master.designated_user_id LIMIT 1) AS seguidor
      FROM (${query} ${whereConstraint.dateRange}) AS master
      LEFT JOIN users ON master.user_id = users.id
      ORDER BY marca_temporal DESC LIMIT ? OFFSET ?`;

    const [rows] = await connection.query(query, whereConstraint.values);

    res.status(200).json({
      pages: totalPages,
      itemsCount: totalItems,
      currentPage: page,
      perPage: limit,
      prevPage: prevPage,
      nextPage: nextPage,
      data: rows,
    });
  } catch (error) {
    console.error("Error al obtener los registros", error);
    res.status(500).json({ error: "Error al obtener los registros" });
  }
});

app.post("/api/rowsPreventa", async (req, res) => {
  try {
    const { id, color, estatusFinalP } = req.body;

    const query =
      "INSERT INTO colorsPreventa (row_id, color, estatus_final) VALUES (?, ?, ?)";
    const values = [id, String(color), estatusFinalP];

    await connection.query(query, values);

    res.status(201).json({ message: "Color guardado exitosamente" });
  } catch (error) {
    console.error("Error al guardar el color", error);
    res.status(500).json({ error: "Error al guardar el color" });
  }
});

app.post("/api/estatusPreventa", async (req, res) => {
  try {
    const { id, estatus } = req.body;

    const query = "UPDATE ip_preventa SET estatus = ? WHERE id = ?";
    const values = [estatus, id];

    await connection.query(query, values);

    res.status(200).json({
      message: "Estatus final actualizado exitosamente",
    });
  } catch (error) {
    console.error("Error al actualizar el estatus final", error);
    res.status(500).json({ error: "Error al actualizar el estatus final" });
  }
});

app.post("/api/preventas/:id/vendedor", async (req, res) => {
  try {
    const id = parseInt(req.params.id);

    if (isNaN(id)) {
      res.status(400).send({ error: "Preventa inválida" });
      return;
    }

    let { userId, estatus } = req.body;
    userId = parseInt(Sanitizer.run(`${userId}`));

    if (isNaN(userId)) {
      res.status(400).send({ error: "Vendedor inválido" });
      return;
    }

    const [ipPreventas] = await connection.query('SELECT id FROM ip_preventa WHERE id = ?', [id]);
    if (ipPreventas.length == 0) {
      res.status(400).send({ error: "Preventa inválida" });
      return;
    }

    const query = "SELECT users.id, roles.value FROM users, roles WHERE users.id = ? AND roles.value = ? AND users.role_id = roles.id";
    const [users] = await connection.query(query, [userId, Role.Preventa]);

    if (users.length == 0) {
      res.status(400).send({ error: "Vendedor inválido" });
      return;
    }

    await connection.query("UPDATE ip_preventa SET user_id = ?, estatus = ? WHERE id = ?", [userId, estatus, id]);

    res.status(201).send({ message: "Vendedor asignado correctamente!" });
  } catch (error) {
    console.error("Error al modificar el registro", error);
    res.status(500).json({ error: "Error al modificar el registro" });
  }
});

app.post("/api/preventas/:id/seguidor", async (req, res) => {
  try {
    const id = parseInt(req.params.id);

    if (isNaN(id)) {
      res.status(400).send({ error: "Preventa inválida" });
      return;
    }

    let { userId } = req.body;
    userId = parseInt(Sanitizer.run(`${userId}`));

    if (isNaN(userId)) {
      res.status(400).send({ error: "Acompañante inválido" });
      return;
    }

    const [ipPreventas] = await connection.query('SELECT id, user_id FROM ip_preventa WHERE id = ?', [id]);
    if (ipPreventas.length == 0) {
      res.status(400).send({ error: "Preventa inválida" });
      return;
    }

    const query = "SELECT users.id, roles.value FROM users, roles WHERE users.id = ? AND roles.value = ? AND users.role_id = roles.id";
    const [users] = await connection.query(query, [userId, Role.Preventa]);

    if (ipPreventas[0].user_id == userId) {
      res.status(400).send({ error: "El acompañante no puede ser el mismo que el vendedor asignado" });
      return;
    }

    if (users.length == 0) {
      res.status(400).send({ error: "Acompañante inválido" });
      return;
    }

    await connection.query("UPDATE ip_preventa SET designated_user_id = ? WHERE id = ?", [userId, id]);

    res.status(201).send({ message: "Acompañante asignado correctamente!" });
  } catch (error) {
    console.error("Error al modificar el registro", error);
    res.status(500).json({ error: "Error al modificar el registro" });
  }
});

// Iniciar el servidor HTTPS
const port = 3300;
const hostname = "easysearchnode.tecnologiaintegrada.mx";

const server = https.createServer(options, app);

server.listen(port, () => {
  console.log(`Servidor HTTPS iniciado en https://${hostname}:${port}/`);
});
