const express = require("express");
const mysql = require("mysql2/promise");
const cors = require("cors");
const { format } = require("date-fns");
const bodyParser = require("body-parser");
const nodemailer = require('nodemailer');
const path = require('path');
const fs = require('fs');
const dayjs = require('dayjs');
const https = require('https');
const moment = require('moment-timezone');
const { Number } = require("twilio/lib/twiml/VoiceResponse");
const Licitacion = require('./src/models/licitacion');
const Sanitizer = require("./src/utils/Sanitizer");
const Role = require("./src/models/Role");
require('dotenv').config({ path: __dirname + '/.env' });

const privateKey = fs.readFileSync(__dirname + "/private.key", "utf-8");
const certificate = fs.readFileSync(__dirname + "/public.cert", "utf-8");

const options = {
  key: privateKey,
  cert: certificate,
};

// Crear la aplicación Express
const app = express();

app.use(bodyParser.urlencoded({ extended: false }));
app.use(bodyParser.json());

// Crear conexión a la base de datos
const connection = mysql.createPool({
  host: process.env.HOSTNAME,
  port: "3306",
  user: process.env.DBUSER,
  password: process.env.PASSWORD,
  database: process.env.DATABASE,
});

// Middleware para habilitar CORS
app.use(cors());

// Middleware para procesar JSON en las solicitudes
app.use(express.json());
// Definir la configuración de la base de datos
const dbConfig = {
  host: process.env.HOSTNAME,
  port: "3306",
  user: process.env.DBUSER,
  password: process.env.PASSWORD,
  database: process.env.DATABASE,
};

// Crear la conexión a la base de datos
const createConnection = async () => {
  const connection = await mysql.createConnection(dbConfig);
  return connection;
};

//Ruta para actualizar los correos
app.put('/actualizar-correo/:correo', async (req, res) => {
  const correoActual = req.params.correo;
  const nuevoCorreo = req.body.nuevo_correo; // Campo esperado para el nuevo correo

  try {
    const connection = await createConnection();

    await connection.execute('UPDATE correos_destino SET correo = ? WHERE correo = ?', [nuevoCorreo, correoActual]);
    res.send('Correo actualizado correctamente');

    connection.end();
  } catch (error) {
    console.error('Error al actualizar el correo en la base de datos', error);
    res.status(500).send('Error al actualizar el correo en la base de datos');
  }
});

// Ruta para eliminar un correo
app.delete('/eliminar-correo/:correo', async (req, res) => {
  const correo = req.params.correo;

  try {
    const connection = await createConnection();

    await connection.execute('DELETE FROM correos_destino WHERE correo = ?', [correo]);
    res.send('Correo eliminado correctamente');

    connection.end();
  } catch (error) {
    console.error('Error al eliminar el correo en la base de datos', error);
    res.status(500).send('Error al eliminar el correo en la base de datos');
  }
});

// Ruta para obtener los correos
app.get('/obtener-correos', async (req, res) => {
  try {
    const connection = await createConnection();

    const [rows] = await connection.execute('SELECT correo FROM correos_destino');
    const correos = rows.map((row) => row.correo);

    res.json(correos);

    connection.end();
  } catch (error) {
    console.error('Error al obtener los correos desde la base de datos', error);
    res.status(500).send('Error al obtener los correos desde la base de datos');
  }
});

// Ruta para enviar correos
app.post('/enviar-correos', async (req, res) => {
  const correosDestino = req.body.correosDestino;
  const mensaje = req.body.mensaje.replace(/\n/g, '<br>');
  const asunto = req.body.asunto;

  const transporter = nodemailer.createTransport({
    host: 'easysearch.tecnologiaintegrada.mx',
    port: 465,
    secure: true,
    auth: {
      user: 'licitaciones@easysearch.tecnologiaintegrada.mx',
      pass: '7vAY5LNS=NL;',
    },
  });

  function enviarCorreo(correoDestino) {
    const mailOptions = {
      from: 'licitaciones@easysearch.tecnologiaintegrada.mx',
      to: correoDestino,
      subject: asunto,
      html: `${mensaje}<br><img src="https://i.postimg.cc/sgxWRXPD/buscador-y-comparador.jpg">`, // Reemplaza URL_DE_LA_IMAGEN con la URL real de la imagen externa
    };

    return new Promise((resolve, reject) => {
      transporter.sendMail(mailOptions, (error, info) => {
        if (error) {
          console.error(error);
          reject(error);
        } else {
          console.log('Correo enviado:', info.response);
          resolve(info.response);
        }
      });
    });
  }

  try {
    const connection = await createConnection();

    const [rows] = await connection.execute('SELECT correo FROM correos_destino');
    const correosRegistrados = rows.map((row) => row.correo);

    const correosParaEnviar = correosDestino.concat(correosRegistrados);

    Promise.all(correosParaEnviar.map(enviarCorreo))
      .then(() => {
        res.send('Correo(s) enviado(s)');
      })
      .catch((error) => {
        console.error('Error al enviar el correo', error);
        res.status(500).send('Error al enviar el correo');
      });

    connection.end();
  } catch (error) {
    console.error('Error al obtener los correos desde la base de datos', error);
    res.status(500).send('Error al obtener los correos desde la base de datos');
  }
});

// Ruta para guardar un correo
app.post('/guardar-correo', async (req, res) => {
  const correo = req.body.correo;

  try {
    const connection = await createConnection();

    await connection.execute('INSERT INTO correos_destino (correo) VALUES (?)', [correo]);
    res.send('Correo guardado correctamente');

    connection.end();
  } catch (error) {
    console.error('Error al guardar el correo en la base de datos', error);
    res.status(500).send('Error al guardar el correo en la base de datos');
  }
});

// Endpoint para enviar un mensaje de correo que se gano la licitación
app.post('/enviar-correos-gano', async (req, res) => {
  const correosDestino = req.body.correosDestino;
  const mensaje = req.body.mensaje.replace(/\n/g, '<br>');
  const asunto = req.body.asunto;

  const transporter = nodemailer.createTransport({
    host: 'easysearch.tecnologiaintegrada.mx',
    port: 465,
    secure: true,
    auth: {
      user: 'licitaciones@easysearch.tecnologiaintegrada.mx',
      pass: '7vAY5LNS=NL;',
    },
  });

  function enviarCorreo(correoDestino) {
    const mailOptions = {
      from: 'licitaciones@easysearch.tecnologiaintegrada.mx',
      to: correoDestino,
      subject: asunto,
      html: `${mensaje}<br><img src="https://i.postimg.cc/sgxWRXPD/buscador-y-comparador.jpg">`, // Reemplaza URL_DE_LA_IMAGEN con la URL real de la imagen externa
    };

    return new Promise((resolve, reject) => {
      transporter.sendMail(mailOptions, (error, info) => {
        if (error) {
          console.error(error);
          reject(error);
        } else {
          console.log('Correo enviado:', info.response);
          resolve(info.response);
        }
      });
    });
  }

  try {
    const connection = await createConnection();

    const [rows] = await connection.execute('SELECT correo FROM correos_destino');
    const correosRegistrados = rows.map((row) => row.correo);

    const correosParaEnviar = [...correosDestino, ...correosRegistrados];

    Promise.all(correosParaEnviar.map(enviarCorreo))
      .then(() => {
        res.send('Correo(s) enviado(s)');
      })
      .catch((error) => {
        console.error('Error al enviar el correo', error);
        res.status(500).send('Error al enviar el correo');
      });

    connection.end();
  } catch (error) {
    console.error('Error al obtener los correos desde la base de datos', error);
    res.status(500).send('Error al obtener los correos desde la base de datos');
  }
});

/// Endpoint para enviar un mensaje de correo que fue desierta la licitación
app.post('/enviar-correos-desierta', (req, res) => {
  const correosDestino = Array.isArray(req.body.correosDestino) ? req.body.correosDestino : [req.body.correosDestino]; // Array de direcciones de correo electrónico
  const mensaje = req.body.mensaje.replace(/\n/g, '<br>'); // Reemplaza los saltos de línea con <br>
  const asunto = req.body.asunto;

  // Configuración del transportador de correo
  const transporter = nodemailer.createTransport({
    host: 'easysearch.tecnologiaintegrada.mx', // Reemplaza con la dirección del servidor SMTP
    port: 465, // Reemplaza con el puerto del servidor SMTP
    secure: true, // Si el servidor utiliza SSL/TLS, cambia a true
    auth: {
      user: 'licitaciones@easysearch.tecnologiaintegrada.mx', // Reemplaza con tu dirección de correo electrónico
      pass: '7vAY5LNS=NL;', // Reemplaza con tu contraseña
    },
  });

  // Función auxiliar para enviar un correo electrónico
  function enviarCorreo(correoDestino) {
    const mailOptions = {
      from: 'licitaciones@easysearch.tecnologiaintegrada.mx',
      to: correoDestino,
      subject: asunto,
      html: `${mensaje}<br><img src="https://i.postimg.cc/sgxWRXPD/buscador-y-comparador.jpg">`, // Reemplaza URL_DE_LA_IMAGEN con la URL real de la imagen externa
    };

    return new Promise((resolve, reject) => {
      // Envío del correo electrónico
      transporter.sendMail(mailOptions, (error, info) => {
        if (error) {
          console.error(error);
          reject(error);
        } else {
          console.log('Correo enviado:', info.response);
          resolve(info.response);
        }
      });
    });
  }

  // Envía el correo electrónico a cada destinatario
  Promise.all(correosDestino.map(enviarCorreo))
    .then(() => {
      res.send('Correo(s) enviado(s)');
    })
    .catch((error) => {
      res.status(500).send('Error al enviar el correo');
    });
});

/// Endpoint para enviar un mensaje de correo que se perdio la licitación
app.post('/enviar-correos-perdio', (req, res) => {
  const correosDestino = Array.isArray(req.body.correosDestino) ? req.body.correosDestino : [req.body.correosDestino]; // Array de direcciones de correo electrónico
  const mensaje = req.body.mensaje.replace(/\n/g, '<br>'); // Reemplaza los saltos de línea con <br>
  const asunto = req.body.asunto;

  // Configuración del transportador de correo
  const transporter = nodemailer.createTransport({
    host: 'easysearch.tecnologiaintegrada.mx', // Reemplaza con la dirección del servidor SMTP
    port: 465, // Reemplaza con el puerto del servidor SMTP
    secure: true, // Si el servidor utiliza SSL/TLS, cambia a true
    auth: {
      user: 'licitaciones@easysearch.tecnologiaintegrada.mx', // Reemplaza con tu dirección de correo electrónico
      pass: '7vAY5LNS=NL;', // Reemplaza con tu contraseña
    },
  });

  // Función auxiliar para enviar un correo electrónico
  function enviarCorreo(correoDestino) {
    const mailOptions = {
      from: 'licitaciones@easysearch.tecnologiaintegrada.mx',
      to: correoDestino,
      subject: asunto,
      html: `${mensaje}<br><img src="https://i.postimg.cc/sgxWRXPD/buscador-y-comparador.jpg">`, // Reemplaza URL_DE_LA_IMAGEN con la URL real de la imagen externa
    };

    return new Promise((resolve, reject) => {
      // Envío del correo electrónico
      transporter.sendMail(mailOptions, (error, info) => {
        if (error) {
          console.error(error);
          reject(error);
        } else {
          console.log('Correo enviado:', info.response);
          resolve(info.response);
        }
      });
    });
  }

  // Envía el correo electrónico a cada destinatario
  Promise.all(correosDestino.map(enviarCorreo))
    .then(() => {
      res.send('Correo(s) enviado(s)');
    })
    .catch((error) => {
      res.status(500).send('Error al enviar el correo');
    });
});

app.get("/api/users", async (req, res) => {
  try {
    let { userRole } = req.query;
    userRole = Sanitizer.run(userRole);

    const query = "SELECT users.id, users.name FROM users, roles WHERE users.role_id = roles.id AND roles.value = ?";
    const [users] = await connection.query(query, [userRole]);

    res.status(200).send({ users: users });
  } catch (error) {
    console.error(error);
    res.status(500).send({ error: "Error al consultar los usuarios" });
  }
});

// Ruta para guardar un nuevo registro
app.post("/api/registros", async (req, res) => {
  try {
    // Obtener los datos del cuerpo de la solicitud
    const {
      usuario,
      nombreLicitacion,
      dependenciaLicitante,
      dependenciaUsuario,
      empresaParticipantes,
      fechaPublicacion,
      dondePublicado,
      fechaEnvioPreguntas,
      horaLimiteEnvioPreguntas,
      comoSeEntregan,
      domicilioCorreo,
      fechaLimiteEntregab,
      fechaJuntaAclaraciones,
      horaRegistroJA,
      horaJA,
      juntaObligatoria,
      domicilioJA,
      presencialOdigital,
      domicilio,
      fechaPresentacion,
      horaRegistroPresentacion,
      horaApertura,
      partidas,
      requierePreventa,
      linkCarpeta,
    } = req.body;

    // Establecer la zona horaria para México
    moment.tz.setDefault('America/Mexico_City'); // Puedes ajustar la zona horaria según sea necesario

    // Obtener la hora actual con la zona horaria configurada
    const marcaTemporal = moment().format('YYYY-MM-DD HH:mm:ss');


    // Consulta SQL para insertar un nuevo registro
    const query = `
      INSERT INTO licitaciones (
        usuario,
        marca_temporal,
        nombre,
        dependencia_licitante,
        dependencia_usuario,
        empresa_participantes,
        fecha_publicacion,
        donde_se_publico,
        fecha_envio_preguntas,
        hora_limite_envio_preguntas,
        como_se_entregan,
        domicilio_correo,
        fecha_limite_entregab,
        fecha_junta_aclaraciones,
        hora_registro_ja,
        hora_ja,
        junta_obligatoria,
        domicilio_ja,
        presencial_digital,
        domicilio,
        fecha_presentacion,
        hora_registro_presentacion,
        hora_apertura,
        partidas,
        requiere_preventa,
        link_carpeta
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    `;

    const values = [
      usuario,
      marcaTemporal,
      nombreLicitacion,
      dependenciaLicitante,
      dependenciaUsuario,
      empresaParticipantes,
      fechaPublicacion,
      dondePublicado,
      fechaEnvioPreguntas,
      horaLimiteEnvioPreguntas,
      comoSeEntregan,
      domicilioCorreo,
      fechaLimiteEntregab,
      fechaJuntaAclaraciones,
      horaRegistroJA,
      horaJA,
      juntaObligatoria,
      domicilioJA,
      presencialOdigital,
      domicilio,
      fechaPresentacion,
      horaRegistroPresentacion,
      horaApertura,
      partidas,
      requierePreventa,
      linkCarpeta
    ];

    // Ejecutar la consulta SQL
    await connection.query(query, values);

    // Enviar respuesta exitosa
    res.status(201).json({ message: "Registro guardado exitosamente" });
  } catch (error) {
    console.error("Error al guardar el registro", error);
    res.status(500).json({ error: "Error al guardar el registro" });
  }
});

app.post("/api/registros/:id/vendedor", async (req, res) => {
  try {
    const id = parseInt(req.params.id);

    if (isNaN(id)) {
      res.status(400).send({ error: "Licitación inválida" });
      return;
    }

    let { userId, estatus } = req.body;
    userId = parseInt(Sanitizer.run(`${userId}`));

    if (isNaN(userId)) {
      res.status(400).send({ error: "Vendedor inválido" });
      return;
    }

    const [bids] = await connection.query('SELECT id FROM licitaciones WHERE id = ?', [id]);
    if (bids.length == 0) {
      res.status(400).send({ error: "Licitación inválida" });
      return;
    }

    const query = "SELECT users.id, roles.value FROM users, roles WHERE users.id = ? AND roles.value = ? AND users.role_id = roles.id";
    const [users] = await connection.query(query, [userId, Role.Preventa]);

    if (users.length == 0) {
      res.status(400).send({ error: "Vendedor inválido" });
      return;
    }

    await connection.query("UPDATE licitaciones SET user_id = ?, estatus_final = ? WHERE id = ?", [userId, estatus, id]);

    res.status(201).send({ message: "Vendedor asignado correctamente!" });
  } catch (error) {
    console.error("Error al modificar el registro", error);
    res.status(500).json({ error: "Error al modificar el registro" });
  }
});

app.post("/api/registros/:id/seguidor", async (req, res) => {
  try {
    const id = parseInt(req.params.id);

    if (isNaN(id)) {
      res.status(400).send({ error: "Licitación inválida" });
      return;
    }

    let { userId } = req.body;
    userId = parseInt(Sanitizer.run(`${userId}`));

    if (isNaN(userId)) {
      res.status(400).send({ error: "Acompañante inválido" });
      return;
    }

    const [bids] = await connection.query('SELECT id, user_id FROM licitaciones WHERE id = ?', [id]);
    if (bids.length == 0) {
      res.status(400).send({ error: "Licitación inválida" });
      return;
    }

    const query = "SELECT users.id, roles.value FROM users, roles WHERE users.id = ? AND roles.value = ? AND users.role_id = roles.id";
    const [users] = await connection.query(query, [userId, Role.Preventa]);

    if (bids[0].user_id == userId) {
      res.status(400).send({ error: "El acompañante no puede ser el mismo que el vendedor asignado" });
      return;
    }

    if (users.length == 0) {
      res.status(400).send({ error: "Acompañante inválido" });
      return;
    }

    await connection.query("UPDATE licitaciones SET designated_user_id = ? WHERE id = ?", [userId, id]);

    res.status(201).send({ message: "Acompañante asignado correctamente!" });
  } catch (error) {
    console.error("Error al modificar el registro", error);
    res.status(500).json({ error: "Error al modificar el registro" });
  }
});

//Ruta para actualizar un registro (una licitación)
app.post('/api/actualizar-registro/:id', async (req, res) => {
  try {
    // Obtener los datos del cuerpo de la solicitud
    const {
      nombreLicitacion,
      dependenciaLicitante,
      dependenciaUsuario,
      empresaParticipantes,
      fechaPublicacion,
      dondePublicado,
      fechaEnvioPreguntas,
      horaLimiteEnvioPreguntas,
      comoSeEntregan,
      domicilioCorreo,
      fechaJuntaAclaraciones,
      horaRegistroJA,
      horaJA,
      juntaObligatoria,
      domicilioJA,
      presencialOdigital,
      domicilio,
      fechaPresentacion,
      horaRegistroPresentacion,
      horaApertura,
      partidas,
      requierePreventa,
      linkCarpeta,
      razonNoPresentacion,
      fechaLimiteEntregab,
    } = req.body;

    // Consulta SQL para insertar un nuevo registro
    const query = `
    UPDATE licitaciones
    SET
      nombre = ?,
      dependencia_licitante = ?,
      dependencia_usuario = ?,
      empresa_participantes = ?,
      fecha_publicacion = ?,
      donde_se_publico = ?,
      fecha_envio_preguntas = ?,
      hora_limite_envio_preguntas = ?,
      como_se_entregan = ?,
      domicilio_correo = ?,
      fecha_junta_aclaraciones = ?,
      hora_registro_ja = ?,
      hora_ja = ?,
      junta_obligatoria = ?,
      domicilio_ja = ?,
      presencial_digital = ?,
      domicilio = ?,
      fecha_presentacion = ?,
      hora_registro_presentacion = ?,
      hora_apertura = ?,
      partidas = ?,
      requiere_preventa = ?,
      link_carpeta = ?,
      razon_no_presentacion = ?,
      fecha_limite_entregab = ?
    WHERE
      id = ?
    `;

    const values = [
      nombreLicitacion,
      dependenciaLicitante,
      dependenciaUsuario,
      empresaParticipantes,
      fechaPublicacion,
      dondePublicado,
      fechaEnvioPreguntas,
      horaLimiteEnvioPreguntas,
      comoSeEntregan,
      domicilioCorreo,
      fechaJuntaAclaraciones,
      horaRegistroJA,
      horaJA,
      juntaObligatoria,
      domicilioJA,
      presencialOdigital,
      domicilio,
      fechaPresentacion,
      horaRegistroPresentacion,
      horaApertura,
      partidas,
      requierePreventa,
      linkCarpeta,
      razonNoPresentacion,
      fechaLimiteEntregab,
      req.params.id
    ];

    // Ejecutar la consulta SQL
    await connection.query(query, values);

    // Enviar respuesta exitosa
    res.status(201).json({ message: "Registro modificado exitosamente" });
  } catch (error) {
    console.error("Error al modificar el registro", error);
    res.status(500).json({ error: "Error al modificar el registro" });
  }
});

const getUniqueValues = async (columnName, alias, req) => {
  let uniqueValues = [];
  try {
    let { page = 1, limit = 20 } = req.query;
    limit = parseInt(limit);
    page = parseInt(page);
    const whereConstraint = Licitacion.whereConstraints(req.query);

    const offset = page <= 1 ? 0 : (limit * (page - 1));

    whereConstraint.values.push(limit, offset);
    let query = `SELECT DISTINCT(master.${columnName}) AS ${alias} FROM (${whereConstraint.query} ${whereConstraint.dateRange}  LIMIT ? OFFSET ?) AS master`;

    const [rows] = await connection.query(query, whereConstraint.values);

    for (const row of rows) {
      if (row[alias] == null || row[alias] == undefined) {
        uniqueValues.push('Vacíos');
      } else {
        uniqueValues.push(row[alias]);
      }
    }
  } catch (error) {
    console.error('getUniqueValues', error);
  }

  return uniqueValues;
}

const getInitialYear = async () => {
  try {
    const query = 'SELECT created_at FROM licitaciones LIMIT 1';
    const [rows] = await connection.query(query);


    const createdAt = rows[0].created_at;
    if (createdAt != undefined || createdAt.length > 10) {
      let initialYear = createdAt.getFullYear();
      const currentYear = new Date();

      if (initialYear == currentYear.getFullYear()) {
        return ['Todos', `${initialYear}`];
      }

      let years = [];
      while (initialYear <= currentYear.getFullYear()) {
        years.push(`${initialYear}`);
        initialYear++;
      }

      years.unshift('Todos')

      return years;
    }
  } catch (error) {
    console.error('getInitialDate', error);
  }

  return [];
}

const getUniqueDates = async (columnName, alias, req, full = false) => {
  let uniqueDates = [];
  try {
    let { page = 1, limit = 20 } = req.query;
    limit = parseInt(limit);
    page = parseInt(page);
    const whereConstraint = Licitacion.whereConstraints(req.query);

    const offset = page <= 1 ? 0 : (limit * (page - 1));

    whereConstraint.values.push(limit, offset);
    let query = `SELECT DISTINCT(LEFT(master.${columnName}, ${full ? 16 : 10})) AS ${alias} FROM (${whereConstraint.query} ${whereConstraint.dateRange}  LIMIT ? OFFSET ?) AS master`;

    const [rows] = await connection.query(query, whereConstraint.values);

    for (const row of rows) {
      if (row[alias] == null || row[alias] == undefined) {
        uniqueDates.push('Vaíos');
      } else {
        if (columnName == "marca_temporal") {
          const dateTime = row[alias].split(" ");
          const date = dateTime[0].split("-");
          const time = dateTime[1];

          uniqueDates.push(`${time} ${date[2]}-${date[1]}-${date[0]}`);
        } else {
          const splitRowValue = row[alias].split('-');
          uniqueDates.push(`${splitRowValue[2]}-${splitRowValue[1]}-${splitRowValue[0]}`);
        }
      }
    }
  } catch (error) {
    console.error('getUniqueDates', error);
  }

  return uniqueDates;
}

// Ruta para obtener todos los registros
app.get("/api/registros", async (req, res) => {
  try {
    let { page = 1, limit = 20 } = req.query;
    limit = parseInt(limit);
    page = parseInt(page);
    const whereConstraint = Licitacion.whereConstraints(req.query);

    const offset = page <= 1 ? 0 : (limit * (page - 1));

    let query = `${whereConstraint.query}`;
    const [items] = await connection.query(`SELECT COUNT(*) AS rows_count FROM (${query} ${whereConstraint.dateRange}) AS filtered_licitaciones`, whereConstraint.values);

    let totalPages = 1;
    const rowsCount = items[0];

    const totalItems = parseInt(rowsCount.rows_count);

    if (totalItems > limit) {
      let doubleRes = totalItems / limit;
      let intResult = Math.round(doubleRes);

      totalPages = intResult > doubleRes ? intResult : intResult + 1;
    }

    const prevPage = totalItems > limit ? (page >= 2 ? page - 1 : null) : null;
    const nextPage = totalItems > limit ? (page < totalPages ? page + 1 : null) : null;

    whereConstraint.values.push(limit, offset);
    query = `
      SELECT master.*, (CASE WHEN master.user_id = users.id
        THEN users.name
        ELSE null
        END) AS vendedor,
        (SELECT users.name AS seguidor FROM users WHERE users.id = master.designated_user_id LIMIT 1) AS seguidor
      FROM (${query} ${whereConstraint.dateRange}) AS master
      LEFT JOIN users ON master.user_id = users.id
      ORDER BY master.created_at DESC`
    query += ' LIMIT ? OFFSET ?';

    // Ejecutar la consulta SQL
    const [rows] = await connection.query(query, whereConstraint.values);

    // Enviar los registros formateados como respuesta
    res.status(200).json({
      pages: totalPages,
      itemsCount: totalItems,
      currentPage: page,
      perPage: limit,
      prevPage: prevPage,
      nextPage: nextPage,
      data: rows,
    });
  } catch (error) {
    console.error("Error al obtener los registros", error);
    res.status(500).json({ error: "Error al obtener los registros" });
  }
});

app.get('/api/registros/filtros', async (req, res) => {
  let filtros = {
    filtroPartidas: [],
    filtroMarcasTemporales: [],
    filtroNombresLicitaciones: [],
    filtroDependenciasLicitantes: [],
    filtroDependenciaUsuarios: [],
    filtroEmpresasParticipantes: [],
    filtroFechasPublicacion: [],
    filtroLugaresPublicacion: [],
    filtroFechasPreguntas: [],
    filtroHorasLimiteEnviop: [],
    filtroTiposDeEntrega: [],
    filtroDomiciliosCorreos: [],
    filtroFechasJuntaAcl: [],
    filtroHorasRegistroJA: [],
    filtroHorasJA: [],
    filtroJuntasObligatorias: [],
    filtroDomiciliosJunta: [],
    filtroTiposJuntas: [],
    filtroDomicilios: [],
    filtroFechasPre: [],
    filtroHorasRegistroPre: [],
    filtroHorasApertura: [],
    filtroRequierePreventa: [],
    filtroPresentacion: [],
    filtroPreguntas: [],
    filtroEstatusFicha: [],
    filtroAnios: [],
  }
  try {
    const filtroPartidas = await getUniqueValues('partidas', 'partida', req);
    const filtroMarcasTemporales = await getUniqueDates('marca_temporal', 'marca', req, true);
    const filtroNombresLicitaciones = await getUniqueValues('nombre', 'nombre', req);
    const filtroDependenciasLicitantes = await getUniqueValues('dependencia_licitante', 'dependencia', req);
    const filtroDependenciaUsuarios = await getUniqueValues('dependencia_usuario', 'usuario', req);
    const filtroEmpresasParticipantes = await getUniqueValues('empresa_participantes', 'empresa', req);
    const filtroFechasPublicacion = await getUniqueDates('fecha_publicacion', 'fecha', req);
    const filtroLugaresPublicacion = await getUniqueValues('donde_se_publico', 'lugar', req);
    const filtroFechasPreguntas = await getUniqueDates('fecha_envio_preguntas', 'fecha', req);
    const filtroHorasLimiteEnviop = await getUniqueValues('hora_limite_envio_preguntas', 'hora', req);
    const filtroTiposDeEntrega = await getUniqueValues('como_se_entregan', 'metodo', req);
    const filtroDomiciliosCorreos = await getUniqueValues('domicilio_correo', 'dc', req);
    const filtroFechasJuntaAcl = await getUniqueDates('fecha_junta_aclaraciones', 'fecha', req);
    const filtroHorasRegistroJA = await getUniqueValues('hora_registro_ja', 'hora', req);
    const filtroHorasJA = await getUniqueValues('hora_ja', 'hora', req);
    const filtroJuntasObligatorias = await getUniqueValues('junta_obligatoria', 'jo', req);
    const filtroDomiciliosJunta = await getUniqueValues('domicilio_ja', 'domicilio', req);
    const filtroTiposJuntas = await getUniqueValues('presencial_digital', 'tipo', req);
    const filtroDomicilios = await getUniqueValues('domicilio', 'domicilio', req);
    const filtroFechasPre = await getUniqueDates('fecha_presentacion', 'fecha', req);
    const filtroHorasRegistroPre = await getUniqueValues('hora_registro_presentacion', 'hora', req);
    const filtroHorasApertura = await getUniqueValues('hora_apertura', 'hora', req);
    const filtroRequierePreventa = await getUniqueValues('requiere_preventa', 'requiere', req);
    const filtroPresentacion = await getUniqueValues('estatus_presentacion', 'presentacion', req);
    const filtroPreguntas = await getUniqueValues('estatus_preguntas', 'estatus_preguntas', req);
    const filtroEstatusFicha = await getUniqueValues('estatus_ficha', 'estatus_ficha', req);
    const initialYear = await getInitialYear();

    filtros.filtroPartidas = filtroPartidas;
    filtros.filtroMarcasTemporales = filtroMarcasTemporales;
    filtros.filtroNombresLicitaciones = filtroNombresLicitaciones;
    filtros.filtroDependenciasLicitantes = filtroDependenciasLicitantes;
    filtros.filtroDependenciaUsuarios = filtroDependenciaUsuarios;
    filtros.filtroEmpresasParticipantes = filtroEmpresasParticipantes;
    filtros.filtroFechasPublicacion = filtroFechasPublicacion;
    filtros.filtroLugaresPublicacion = filtroLugaresPublicacion;
    filtros.filtroFechasPreguntas = filtroFechasPreguntas;
    filtros.filtroHorasLimiteEnviop = filtroHorasLimiteEnviop;
    filtros.filtroTiposDeEntrega = filtroTiposDeEntrega;
    filtros.filtroDomiciliosCorreos = filtroDomiciliosCorreos;
    filtros.filtroFechasJuntaAcl = filtroFechasJuntaAcl;
    filtros.filtroHorasRegistroJA = filtroHorasRegistroJA;
    filtros.filtroHorasJA = filtroHorasJA;
    filtros.filtroJuntasObligatorias = filtroJuntasObligatorias;
    filtros.filtroDomiciliosJunta = filtroDomiciliosJunta;
    filtros.filtroTiposJuntas = filtroTiposJuntas;
    filtros.filtroDomicilios = filtroDomicilios;
    filtros.filtroFechasPre = filtroFechasPre;
    filtros.filtroHorasRegistroPre = filtroHorasRegistroPre;
    filtros.filtroHorasApertura = filtroHorasApertura;
    filtros.filtroRequierePreventa = filtroRequierePreventa;
    filtros.filtroPresentacion = filtroPresentacion;
    filtros.filtroPreguntas = filtroPreguntas;
    filtros.filtroEstatusFicha = filtroEstatusFicha;
    filtros.filtroAnios = initialYear;

    res.status(200).json(filtros);
  } catch (error) {
    console.error('Error al obtener los filtros', error);
    res.status(500).json({ error: 'Error al obtener los filtros' });
  }
});

app.post("/api/estatus/observaciones", async (req, res) => {
  try {
    const { id, estatus } = req.body;

    const query =
      "UPDATE licitaciones SET observaciones = ? WHERE id = ?";
    const values = [estatus, id];

    await connection.query(query, values);

    res.status(200).json({ message: "Estatus de preguntas actualizado exitosamente" });
  } catch (error) {
    console.error("Error al actualizar el estatus de preguntas", error);
    res.status(500).json({ error: "Error al actualizar el estatus de preguntas" });
  }
});

//Ruta para agregar razon de no presentación
app.post("/api/registro/razon-no-presentacion", async (req, res) => {
  try {
    const { id, razonNoP } = req.body;

    const query =
      "UPDATE licitaciones SET razon_no_presentacion = ? WHERE id = ?";
    const values = [razonNoP, id];

    await connection.query(query, values);

    res.status(200).json({ message: "Razón de no presentación actualizado exitosamente" });
  } catch (error) {
    console.error("Error al actualizar la razón de no presentación", error);
    res.status(500).json({ error: "Error al actualizar la razón de no presentación" });
  }
});

//Ruta para modificar fecha de limite de entrega de bienes
app.post("/api/registro/fecha-limite-bienes", async (req, res) => {
  try {
    const { id, fechaLBienes } = req.body;

    const query =
      "UPDATE licitaciones SET fecha_limite_entregab = ? WHERE id = ?";
    const values = [fechaLBienes, id];

    await connection.query(query, values);

    res.status(200).json({ message: "Fecha limite de entrega de bienes actualizado exitosamente" });
  } catch (error) {
    console.error("Error al actualizar la fecha limite de entrega de bienes", error);
    res.status(500).json({ error: "Error al actualizar la fecha limite de entrega de bienes" });
  }
});

app.post("/api/estatus/preguntas", async (req, res) => {
  try {
    const { id, estatus, usuarioPreguntas } = req.body; // Agregar "usuarioPreguntas" al destructuring

    const fechaHora = dayjs(); // Obtenemos la fecha y hora actual con dayjs
    const formattedFechaHora = fechaHora.format('YYYY-MM-DD HH:mm:ss'); // Formato para columna de tipo DATETIME

    const query =
      "UPDATE licitaciones SET estatus_preguntas = ?, fecha_hora_guardado_preguntas = ?, usuario_preguntas = ? WHERE id = ?"; // Actualizar la consulta SQL con el nuevo campo "usuario_preguntas"
    const values = [estatus, formattedFechaHora, usuarioPreguntas, id]; // Incluir "usuarioPreguntas" en los valores

    await connection.query(query, values);

    res.status(200).json({ message: "Estatus de preguntas actualizado exitosamente" });
  } catch (error) {
    console.error("Error al actualizar el estatus de preguntas", error);
    res.status(500).json({ error: "Error al actualizar el estatus de preguntas" });
  }
});


// Ruta para guardar el estatus de la junta
app.post("/api/estatus/junta", async (req, res) => {
  try {
    const { id, estatus, usuarioJunta } = req.body;
    const fechaHora = dayjs();

    const formattedFechaHora = fechaHora.format('YYYY-MM-DD HH:mm:ss');

    const query = "UPDATE licitaciones SET estatus_junta = ?, fecha_hora_guardado_junta = ?, usuario_junta = ? WHERE id = ?";
    const values = [estatus, formattedFechaHora, usuarioJunta, id];

    await connection.query(query, values);

    res
      .status(200)
      .json({ message: "Estatus de junta actualizado exitosamente" });
  } catch (error) {
    console.error("Error al actualizar el estatus de junta", error);
    res.status(500).json({ error: "Error al actualizar el estatus de junta" });
  }
});

// Ruta para guardar el estatus de la presentación
app.post("/api/estatus/presentacion", async (req, res) => {
  try {
    const { id, estatus, usuarioPresentacion } = req.body;
    const fechaHora = dayjs();

    const formattedFechaHora = fechaHora.format('YYYY-MM-DD HH:mm:ss');

    const query =
      "UPDATE licitaciones SET estatus_presentacion = ?, fecha_hora_guardado_presentacion = ?, usuario_presentacion = ? WHERE id = ?";
    const values = [estatus, formattedFechaHora, usuarioPresentacion, id];

    await connection.query(query, values);

    res
      .status(200)
      .json({ message: "Estatus de presentación actualizado exitosamente" });
  } catch (error) {
    console.error("Error al actualizar el estatus de presentación", error);
    res
      .status(500)
      .json({ error: "Error al actualizar el estatus de presentación" });
  }
});

// Ruta para guardar el estatus de la ficha tecnica
app.post("/api/estatus/ficha", async (req, res) => {
  try {
    const { id, estatus, usuarioFicha } = req.body; // Agregar "usuarioPreguntas" al destructuring

    const fechaHora = dayjs(); // Obtenemos la fecha y hora actual con dayjs
    const formattedFechaHora = fechaHora.format('YYYY-MM-DD HH:mm:ss'); // Formato para columna de tipo DATETIME

    const query =
      "UPDATE licitaciones SET estatus_ficha = ?, fecha_hora_guardado_ficha = ?, usuario_ficha = ? WHERE id = ?"; // Actualizar la consulta SQL con el nuevo campo "usuario_preguntas"
    const values = [estatus, formattedFechaHora, usuarioFicha, id]; // Incluir "usuarioPreguntas" en los valores

    await connection.query(query, values);

    res.status(200).json({ message: "Estatus de ficha actualizado exitosamente" });
  } catch (error) {
    console.error("Error al actualizar el estatus de ficha", error);
    res.status(500).json({ error: "Error al actualizar el estatus de preguntas" });
  }
});

// Ruta para obtener los valores de estatus de un registro específico
app.get("/api/estatus/:id", async (req, res) => {
  try {
    const { id } = req.params;

    const query =
      "SELECT observaciones, estatus_preguntas, estatus_junta, estatus_presentacion, estatus_ficha FROM licitaciones WHERE id = ?";
    const values = [id];

    const [rows] = await connection.query(query, values);

    const estatus = {
      observaciones: rows[0].observaciones,
      estatusPreguntas: rows[0].estatus_preguntas,
      estatusJunta: rows[0].estatus_junta,
      estatusPresentacion: rows[0].estatus_presentacion,
      estatusFicha: rows[0].estatus_ficha,
    };

    res.status(200).json(estatus);
  } catch (error) {
    console.error("Error al obtener los valores de estatus", error);
    res.status(500).json({ error: "Error al obtener los valores de estatus" });
  }
});

// Ruta para guardar el estatus final
app.post("/api/estatus/final", async (req, res) => {
  try {
    const { id, estatus, usuarioFinal } = req.body;
    const fechaHora = dayjs();

    const formattedFechaHora = fechaHora.format('YYYY-MM-DD HH:mm:ss');

    const query = "UPDATE licitaciones SET estatus_final = ?, fecha_hora_guardado_final = ?, usuario_final = ? WHERE id = ?";
    const values = [estatus, formattedFechaHora, usuarioFinal, id];

    await connection.query(query, values);

    res.status(200).json({ message: "Estatus final actualizado exitosamente" });
  } catch (error) {
    console.error("Error al actualizar el estatus final", error);
    res.status(500).json({ error: "Error al actualizar el estatus final" });
  }
});

// Ruta para obtener el estatus final de un registro
app.get("/api/estatus/final/:id", async (req, res) => {
  try {
    const { id } = req.params;

    const query = "SELECT estatus_final FROM licitaciones WHERE id = ?";
    const values = [id];

    const [rows] = await connection.query(query, values);

    const estatusFinal = rows[0].estatus_final;

    res.status(200).json({ estatusFinal });
  } catch (error) {
    console.error("Error al obtener el estatus final", error);
    res.status(500).json({ error: "Error al obtener el estatus final" });
  }
});

// Ruta para guardar el estado de deshabilitación
app.post("/api/estatus/deshabilitar", async (req, res) => {
  try {
    const { id, deshabilitado } = req.body;

    const query =
      "UPDATE licitaciones SET deshabilitado = ? WHERE id = ?";
    const values = [deshabilitado, id];

    await connection.query(query, values);

    res.status(200).json({ message: "Estado de deshabilitación actualizado exitosamente" });
  } catch (error) {
    console.error("Error al actualizar el estado de deshabilitación", error);
    res.status(500).json({ error: "Error al actualizar el estado de deshabilitación" });
  }
});

// Ruta para obtener el estado de deshabilitación de un registro
app.get("/api/estatus/deshabilitar/:id", async (req, res) => {
  try {
    const { id } = req.params;

    const query = "SELECT deshabilitado FROM licitaciones WHERE id = ?";
    const values = [id];

    const [rows] = await connection.query(query, values);

    const deshabilitado = rows[0].deshabilitado;

    res.status(200).json({ deshabilitado });
  } catch (error) {
    console.error("Error al obtener el estado de deshabilitación", error);
    res.status(500).json({ error: "Error al obtener el estado de deshabilitación" });
  }
});

app.post("/api/:id/product-status", async (req, res) => {
  try {
    const { id } = req.params;
    const { status } = req.body;

    const [licitaciones] = await connection.query('SELECT id FROM licitaciones WHERE id = ?', [id]);
    if (licitaciones.length == 0) {
      res.status(400).send({ error: "Licitación inválida" });
      return;
    }

    if (status == null || status === undefined) {
      res.status(400).send({ error: "El campo status es requerido" });
      return;
    }

    const query = "UPDATE licitaciones set estatus_productos = ? WHERE id = ?";

    await connection.query(query, [status, id]);

    res.status(200).send({ message: "Estatus actualizado correctamente" });
  } catch (error) {
    console.error("Error al actualizar el estatus de los productos", error);
    res.status(500).json({ error: "Error al actualizar el estatu de los productos" });
  }
});

app.post("/api/rows", async (req, res) => {
  try {
    const { id, color, estatusFinal } = req.body;

    const query =
      "INSERT INTO colors (row_id, color, estatus_final) VALUES (?, ?, ?)";
    const values = [id, String(color), estatusFinal];

    await connection.query(query, values);

    res.status(201).json({ message: "Color guardado exitosamente" });
  } catch (error) {
    console.error("Error al guardar el color", error);
    res.status(500).json({ error: "Error al guardar el color" });
  }
});

// Ruta para obtener los colores de las filas
app.get("/api/rows/colors", async (req, res) => {
  try {
    const query = "SELECT row_id, color FROM colors";

    const [rows] = await connection.query(query);

    const colors = rows.reduce((acc, row) => {
      acc[row.row_id] = row.color;
      return acc;
    }, {});

    res.status(200).json(colors);
  } catch (error) {
    console.error("Error al obtener los colores de las filas", error);
    res
      .status(500)
      .json({ error: "Error al obtener los colores de las filas" });
  }
});

// Iniciar el servidor HTTPS
const port = 3200;
const hostname = 'easysearchnode.tecnologiaintegrada.mx';

const server = https.createServer(options, app);

server.listen(port, () => {
  console.log(`Servidor HTTPS iniciado en https://${hostname}:${port}/`);
});
